<?php
/**
 * send_batch.php
 * Cron worker to process queued emails (with retry logic)
 *
 * Run via CLI:
 * php /path/to/send_batch.php
 * 
 * Or set up cron (runs every minute):
 * * * * * php /home/username/public_html/send_batch.php >> /home/username/send_log.txt 2>&1
 */

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error_log.txt');

require __DIR__ . '/vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/* ---------- DATABASE CONFIG ---------- */
const DB_HOST = 'localhost';
$dsn  = 'mysql:host=localhost;dbname=mfapqktjlugecwn_rsvp;charset=utf8mb4';
$user = 'mfapqktjlugecwn_rsvp';
$pass = 'ZankuMania';

/* ---------- BATCH SETTINGS ---------- */
$batchSize   = 10;  // Number of emails to send per run (tune as needed)
$maxAttempts = 3;   // Retry up to 3 times before marking failed

try {
    $pdo = new PDO($dsn, $user, $pass, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
} catch (PDOException $e) {
    error_log("❌ Database Connection Failed: " . $e->getMessage());
    exit("Database connection error\n");
}

/* ---------- FETCH PENDING JOBS ---------- */
try {
    $pdo->beginTransaction();

    $stmt = $pdo->prepare("
        SELECT * FROM email_queue
        WHERE status='pending'
           OR (status='failed' AND COALESCE(attempts,0) < :max)
        ORDER BY id ASC
        LIMIT :limit
        FOR UPDATE
    ");
    $stmt->bindValue(':max', $maxAttempts, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $batchSize, PDO::PARAM_INT);
    $stmt->execute();
    $jobs = $stmt->fetchAll();

    $pdo->commit();
} catch (Exception $e) {
    error_log("❌ Queue fetch failed: " . $e->getMessage());
    $pdo->rollBack();
    exit;
}

if (!$jobs) {
    echo "[" . date('Y-m-d H:i:s') . "] No jobs to process.\n";
    exit;
}

/* ---------- PROCESS EACH JOB ---------- */
foreach ($jobs as $job) {
    $mail = new PHPMailer(true);

    try {
        $mail->isSMTP();
        $mail->Host       = $job['smtp_host'];
        $mail->SMTPAuth   = true;
        $mail->Username   = $job['smtp_user'];
        $mail->Password   = $job['smtp_pass'];
        $mail->Port       = $job['smtp_port'];
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS
                            ? PHPMailer::ENCRYPTION_SMTPS
                            : PHPMailer::ENCRYPTION_STARTTLS;

        $mail->Priority   = 1;
        $mail->addCustomHeader('X-Priority', '1 (Highest)');
        $mail->addCustomHeader('Importance', 'High');

        $mail->setFrom($job['organizer'] ?: $job['smtp_user'], $job['from_name']);
        $mail->addAddress($job['recipient_email']);
        $mail->isHTML(true);

        // Randomized Subject
        $randSubject = strtoupper(bin2hex(random_bytes(5)));
        $mail->Subject = ($job['event_title'] ?? 'Invitation') . " #" . $randSubject;

        // Replace placeholders dynamically
        $body = $job['message_body'];
        $body = str_replace('[email]', $job['recipient_email'], $body);
        $body = str_replace('[randompath]', random_int(100000000, 9999999999), $body);
        $body = str_replace('[randomsubdomain]', substr(md5(uniqid('', true)), 0, 10), $body);

        $mail->Body    = $body;
        $mail->AltBody = strip_tags($body);

        // Attachments
        $attachments = json_decode($job['attachments'], false) ?: [];
        foreach ($attachments as $path) {
            if (file_exists($path)) {
                $info = pathinfo($path);
                $suffix = bin2hex(random_bytes(4));
                $newName = $info['filename'] . '_' . $suffix .
                           (!empty($info['extension']) ? '.' . $info['extension'] : '');
                $mail->addAttachment($path, $newName);
            }
        }

        // Send Email
        $mail->send();

        $updateStmt = $pdo->prepare("UPDATE email_queue
            SET status='sent', sent_at=NOW()
            WHERE id=?");
        $updateStmt->execute([$job['id']]);

        echo "✅ [" . date('H:i:s') . "] Sent to {$job['recipient_email']}\n";

    } catch (Exception $e) {
        $updateStmt = $pdo->prepare("
            UPDATE email_queue
            SET status='failed',
                last_error=?,
                attempts=COALESCE(attempts,0)+1
            WHERE id=?");
        $updateStmt->execute([$e->getMessage(), $job['id']]);

        error_log("❌ [" . date('H:i:s') . "] Failed to {$job['recipient_email']}: {$e->getMessage()}");
    }
}
echo "[" . date('Y-m-d H:i:s') . "] Batch processing complete.\n";