<?php
declare(strict_types=1);

require __DIR__ . '/vendor/autoload.php'; // Composer autoload

ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/error_log.txt');
error_reporting(E_ALL);
ini_set('display_errors', '0');

/* --------------------------------------------------------
   SMTP / FORM INPUTS
---------------------------------------------------------*/
$smtpHost    = trim($_POST['smtp_host'] ?? '');
$smtpPort    = (int)($_POST['smtp_port'] ?? 465);
$smtpSecure  = $_POST['smtp_secure'] ?? 'ssl';
$smtpUser    = trim($_POST['smtp_user'] ?? '');
$smtpPass    = $_POST['smtp_pass'] ?? '';

$organizer   = trim($_POST['organizer'] ?? '');
$fromName    = trim($_POST['from_name'] ?? 'Event Manager');
$eventTitle  = trim($_POST['event_title'] ?? 'Untitled Event');
$eventDesc   = $_POST['event_desc'] ?? '';
$eventLoc    = $_POST['event_loc'] ?? '';
$eventStart  = $_POST['event_start'] ?? null;
$eventEnd    = $_POST['event_end'] ?? null;
$messageBody = $_POST['message_body'] ?? '';

/* --------------------------------------------------------
   HELPER FUNCTIONS
---------------------------------------------------------*/
function randStr(int $len = 6): string {
    $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    $out = '';
    $max = strlen($chars) - 1;
    for ($i = 0; $i < $len; $i++) $out .= $chars[random_int(0, $max)];
    return $out;
}

/* --------------------------------------------------------
   CSV RECIPIENT IMPORT
---------------------------------------------------------*/
$csvFile = (isset($_FILES['recipients']) && $_FILES['recipients']['error'] === UPLOAD_ERR_OK)
    ? $_FILES['recipients']['tmp_name']
    : __DIR__ . '/recipients.csv';

$recipients = [];
if (($h = fopen($csvFile, 'r')) !== false) {
    while (($row = fgetcsv($h, 1000, ',')) !== false) {
        $email = trim($row[0] ?? '');
        if ($email && filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $recipients[] = $email;
        }
    }
    fclose($h);
}
if (empty($recipients)) {
    exit('<p style="color:red;font-family:sans-serif;">❌ No valid recipients found in CSV.</p>');
}

/* --------------------------------------------------------
   HANDLE ATTACHMENTS (SAVE WITH RANDOM SUFFIX AT BACK)
---------------------------------------------------------*/
$uploadDir = __DIR__ . '/uploads';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0775, true);
}

$storedFiles = [];
if (!empty($_FILES['attachments']['name'][0])) {
    foreach ($_FILES['attachments']['name'] as $i => $origName) {
        if ($_FILES['attachments']['error'][$i] !== UPLOAD_ERR_OK) continue;

        $tmpPath = $_FILES['attachments']['tmp_name'][$i];
        $info = pathinfo($origName);
        $ext = isset($info['extension']) && $info['extension'] !== '' ? '.' . $info['extension'] : '';
        $base = $info['filename'];

        // Append random suffix (at the BACK before extension)
        $suffix = randStr(6);
        $newName = $base . '_' . $suffix . $ext;
        $dest = $uploadDir . DIRECTORY_SEPARATOR . $newName;

        if (move_uploaded_file($tmpPath, $dest)) {
            // store relative path only (uploads/filename)
            $storedFiles[] = 'uploads/' . $newName;
        }
    }
}
$attachmentsJson = $storedFiles ? json_encode($storedFiles, JSON_UNESCAPED_SLASHES) : null;

/* --------------------------------------------------------
   DATABASE CONNECTION
---------------------------------------------------------*/
$dsn  = 'mysql:host=localhost;dbname=mfapqktjlugecwn_rsvp;charset=utf8mb4';
$dbUser = 'mfapqktjlugecwn_rsvp';
$dbPass = 'ZankuMania';

try {
    $pdo = new PDO($dsn, $dbUser, $dbPass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);
} catch (PDOException $e) {
    exit('<p style="color:red">❌ DB Connection Failed:</p> ' . htmlspecialchars($e->getMessage()));
}

/* --------------------------------------------------------
   INSERT INTO QUEUE
---------------------------------------------------------*/
$sql = "INSERT INTO email_queue
(recipient_email, message_body, attachments, smtp_host, smtp_port, smtp_secure,
 smtp_user, smtp_pass, organizer, from_name, event_title, event_desc, event_loc, event_start, event_end, status)
VALUES
(:recipient_email, :message_body, :attachments, :smtp_host, :smtp_port, :smtp_secure,
 :smtp_user, :smtp_pass, :organizer, :from_name, :event_title, :event_desc, :event_loc, :event_start, :event_end, 'pending')";

$stmt = $pdo->prepare($sql);

$ok = $fail = 0;
foreach ($recipients as $email) {
    try {
        $stmt->execute([
            ':recipient_email' => $email,
            ':message_body'    => $messageBody,
            ':attachments'     => $attachmentsJson,
            ':smtp_host'       => $smtpHost,
            ':smtp_port'       => $smtpPort,
            ':smtp_secure'     => $smtpSecure,
            ':smtp_user'       => $smtpUser,
            ':smtp_pass'       => $smtpPass,
            ':organizer'       => $organizer,
            ':from_name'       => $fromName,
            ':event_title'     => $eventTitle,
            ':event_desc'      => $eventDesc,
            ':event_loc'       => $eventLoc,
            ':event_start'     => $eventStart ?: null,
            ':event_end'       => $eventEnd ?: null
        ]);
        $ok++;
    } catch (Exception $e) {
        error_log('Queue insert error: ' . $e->getMessage());
        $fail++;
    }
}

/* --------------------------------------------------------
   RESPONSE OUTPUT
---------------------------------------------------------*/
echo '<!doctype html><html><head><meta charset="utf-8"><title>Queue Result</title></head>';
echo '<body style="font-family:Arial,Helvetica,sans-serif;padding:20px">';
echo "<h2>📬 Bulk RSVP Mailer</h2>";
echo "<p><strong>Total Recipients:</strong> " . count($recipients) . "</p>";
echo "<p style='color:green'>✅ Successful inserts: $ok</p>";
echo "<p style='color:red'>❌ Failed inserts: $fail</p>";
echo "<p>Attachments stored in <code>/uploads/</code> and linked in DB as JSON.</p>";
echo "<p><strong>Worker:</strong> Run <code>php workers.php</code> via cron or CLI to process the queue.</p>";
echo '</body></html>';
